﻿

Function Convert-etcToBinOrtxt {


[CmdletBinding()]
param(
    [Parameter(Mandatory=$true)][string]$InputFileOrFolder,
    [Parameter(Mandatory=$false)][string]$OutputFolder = "",
    [string]$InputPattern     = "*.etc"
    , [switch]$Recurse
    , [switch]$Force


)

#Set-StrictMode -Version Latest
$ErrorActionPreference = 'Stop'

Function Convert-etcToBin {
  param(
    [Parameter(Mandatory=$true)][string]$inPath
    ,[Parameter(Mandatory=$true)][string]$outPath
    ,[switch]$Force
  )
    
    if ((Test-Path -LiteralPath $outPath -PathType Leaf) -and (-not $Force)) {
    write-host "Output file exists, use -Force to overwrite: $OutPath"
    Return "Output file exists, use -Force to overwrite: $OutPath"
    }


    Write-Host "`nConverting:`n  IN : $inPath`n  OUT: $outPath"
    $inFs  = [System.IO.File]::Open($inPath, [System.IO.FileMode]::Open, [System.IO.FileAccess]::Read, [System.IO.FileShare]::Read)
    $outFs = [System.IO.File]::Open($outPath, [System.IO.FileMode]::Create, [System.IO.FileAccess]::Write, [System.IO.FileShare]::None)
    [int]$EnoughSectorSize     = 2352 + 96
    [int]$qsoSectorSize       = 2352 
    [int]$BlockSectors        = 256

    try {
        $totalLen = $inFs.Length
        

        #  Trim per sector (EnoughSectorSize -> qsoSectorSize)
        if ($EnoughSectorSize -le 0 -or $qsoSectorSize -le 0 -or $qsoSectorSize -gt $EnoughSectorSize) {
            throw "Invalid sector sizing. EnoughSectorSize must be >= qsoSectorSize and > 0."
        }

        $payloadBytes     = $totalLen 
        $totalSectors     = [math]::Floor($payloadBytes / $EnoughSectorSize)
        $partialBytesTail = $payloadBytes % $EnoughSectorSize

        if ($partialBytesTail -ne 0) {
            Write-Warning "Input payload is not a whole number of Enough sectors. Ignoring last $partialBytesTail trailing bytes."
        }

        #Write-Host " - Mode: TrimPerSector"
        #Write-Host "   EnoughSectorSize: $EnoughSectorSize bytes; qsoSectorSize: $qsoSectorSize bytes"
        #Write-Host "   Payload bytes: $payloadBytes; Sectors: $totalSectors"

        $sectorsPerBlock = [math]::Max(1, $BlockSectors)
        $readBlockBytes  = $sectorsPerBlock * $EnoughSectorSize
        $blockBuf        = New-Object byte[] $readBlockBytes

        $sectorsRemaining = $totalSectors
        while ($sectorsRemaining -gt 0) {
            $thisBlockSectors = [math]::Min($sectorsPerBlock, $sectorsRemaining)
            $thisBlockBytes   = $thisBlockSectors * $EnoughSectorSize

            $read = 0
            while ($read -lt $thisBlockBytes) {
                $chunk = $inFs.Read($blockBuf, $read, $thisBlockBytes - $read)
                if ($chunk -le 0) { break }
                $read += $chunk
            }
            if ($read -lt $thisBlockBytes) { break }

            # Write only the qso portion of each sector
            for ($s = 0; $s -lt $thisBlockSectors; $s++) {
                $offset = $s * $EnoughSectorSize
                $outFs.Write($blockBuf, $offset, $qsoSectorSize)
            }

            $sectorsRemaining -= $thisBlockSectors
        }

        $expectedqsoBytes = ($totalSectors * $qsoSectorSize)
        $actualqsoBytes   = $outFs.Length
        Write-Host " - Done. Output size: $actualqsoBytes bytes (expected ≈ $expectedqsoBytes)"
        
    }
    finally {
        $outFs.Flush(); $outFs.Dispose()
        $inFs.Dispose()
    }
} # End of conversion etc to BIN

Function Convert-etcTotxt {
    param(
    [Parameter(Mandatory = $true)]
    [ValidateNotNullOrEmpty()]
    [string]$InputPath,

    [Parameter(Mandatory = $false)]
    [string]$OutputPath

    # , [ValidateSet('ASCII','Latin1','UTF8')]
    # [string]$TextEncoding = 'Latin1',

    , [switch]$Force

    # [switch]$Sanitize
    )

    #----------------------------- Helpers -----------------------------

    function Get-TextEncoding {
    param([string]$Name = "UTF8")
    switch ($Name) {
        'ASCII'  { return [System.Text.Encoding]::ASCII }
        'Latin1' { return [System.Text.Encoding]::GetEncoding(28591) } # qso-8859-1
        'UTF8'   { return [System.Text.Encoding]::UTF8 }
        default  { return [System.Text.Encoding]::GetEncoding(28591) }
    }
    }

    function Find-DelimiterIndex {
    param([byte[]]$Buffer, [int]$StartIndex)
    $len = $Buffer.Length
    for ($i = [Math]::Max(0,$StartIndex); $i -le $len - 3; $i++) {
        if ($Buffer[$i] -eq 1 -and $Buffer[$i+1] -eq 16 -and $Buffer[$i+2] -eq 0) {
        return $i
        }
    }
    return -1
    }

    function Read-HeaderRecord {
    param([byte[]]$Buf, [int]$Index)
    if ($Index + 11 -gt $Buf.Length) { return $null }
    return ,@(
        $Buf[$Index + 0], # B1 (should be 1)
        $Buf[$Index + 1], # B2 (should be 16)
        $Buf[$Index + 2], # B3 (should be 0)
        $Buf[$Index + 3], # B4 (record code or mylane number)
        $Buf[$Index + 4],
        $Buf[$Index + 5],
        $Buf[$Index + 6],
        $Buf[$Index + 7],
        $Buf[$Index + 8], # B9  (mm in mylane records; elsewhere may hold other values)
        $Buf[$Index + 9], # B10 (ss in mylane records)
        $Buf[$Index + 10] # B11 (ff in mylane records)
    )
    }

    function Decode-BytesToString {
    param(
        [byte[]]$Bytes,
        [System.Text.Encoding]$Encoding,
        [switch]$Sanitize
    )
    if (-not $Bytes -or $Bytes.Length -eq 0) { return '' }
    $str = $Encoding.GetString($Bytes)
    # Remove embedded nulls
    $str = $str -replace "`0",""
    if ($Sanitize) {
        # Strip non-printable (keep ASCII 0x20–0x7E)
        $sb = New-Object System.Text.StringBuilder
        foreach ($ch in $str.ToCharArray()) {
        if ([int][char]$ch -ge 32 -and [int][char]$ch -le 126) { [void]$sb.Append($ch) }
        }
        $str = $sb.ToString()
        $str = ($str -replace '\s+',' ').Trim()
    }
    return $str.Trim()
    }

    # Process a 12-byte text payload:
    # Start at initial tno; on each 0x00, increment target (wholecollection=0 -> mylane1).
    function Process-Payload {
        param(
            [byte[]]$Payload,    # 12 bytes: bytes 5..16 of the 18-byte record
            [int]$Startmylane,    # byte 2 of the record
            [System.Collections.Generic.List[byte]]$wholecollectionBuf, # wholecollection (mylane 0) collector
            [hashtable]$mylaneBufMap # mylaneNo -> List[byte]
        )
        $t = [int]$Startmylane
        foreach ($b in $Payload) {
            if ($b -eq 0) { $t++; continue }
            if ($t -eq 0) {
            $wholecollectionBuf.Add([byte]$b)
            } else {
            if (-not $mylaneBufMap.ContainsKey($t)) {
                $mylaneBufMap[$t] = New-Object System.Collections.Generic.List[byte]
            }
            $mylaneBufMap[$t].Add([byte]$b)
            }
        }
    }

    # Re-synchronize scanning for 18-byte body records:
    # Find a byte 0x80 or 0x81, ensure next 12 payload bytes are plausibly text/zero.
    function Scan-BodyRecords {
    param([byte[]]$Buf, [int]$StartIndex)
    $records = New-Object System.Collections.Generic.List[object]
    $i = $StartIndex
    while ($i -le $Buf.Length - 18) {
        $b0 = [int]$Buf[$i]
        $b1 = [int]$Buf[$i+1]
        if (($b0 -eq 0x80 -or $b0 -eq 0x81) -and $b1 -ge 0 -and $b1 -le 99) {
        $payload = $Buf[($i+4)..($i+15)]
        # Heuristic: at least 8/12 bytes are printable ASCII (32..126) or zero
        $printables = 0
        foreach ($v in $payload) {
            if (($v -ge 32 -and $v -le 126) -or $v -eq 0) { $printables++ }
        }
        if ($printables -ge 8) {
            $records.Add([pscustomobject]@{
            Offset  = $i
            Type    = $b0    # 0x80=Title, 0x81=author
            mylaneNo = $b1
            Data    = $payload
            })
            $i += 18
            continue
        }
        }
        $i++
    }
    return $records
    }

    #----------------------------- Load file -----------------------------

    if (-not (Test-Path -LiteralPath $InputPath -PathType Leaf)) {
    throw "Input file not found: $InputPath"
    }
    if ((Test-Path -LiteralPath $outputPath -PathType Leaf) -and (-not $Force)) {
    write-host "Output file exists, use -Force to overwrite: $OutputPath"
    Return "Output file exists, use -Force to overwrite: $OutputPath"
    }
    [byte[]]$buf = [System.IO.File]::ReadAllBytes((Resolve-Path -LiteralPath $InputPath))
    $enc = Get-TextEncoding -Name $TextEncoding

    if (-not $OutputPath -or [string]::IsNullOrWhiteSpace($OutputPath)) {
        $OutputPath = [System.IO.Path]::Combine(
            (Split-Path -Path $InputPath -Parent),
            ([System.IO.Path]::GetFileNameWithoutExtension($InputPath) + '.txt')
        )
    }

    #----------------------------- Parse header -----------------------------

    # Collect every 11-byte record starting at each 0x01 0x10 0x00 delimiter
    $hdrIdx = Find-DelimiterIndex -Buffer $buf -StartIndex 0
    $hdrRecs = @()
    while ($hdrIdx -ge 0 -and $hdrIdx + 11 -le $buf.Length) {
    $hdrRecs += ,@($hdrIdx, (Read-HeaderRecord -Buf $buf -Index $hdrIdx))
    $hdrIdx = Find-DelimiterIndex -Buffer $buf -StartIndex ($hdrIdx + 3)
    }

    # Iterate header records; mylane records begin when B4 is in 1..99
    $mylaneTimes = @{} # mylaneNo -> @{mm; ss; ff}
    $headerEnd = $null
    $startedmylanes = $false
    [int]$nummylanes = 0
    $discTotal = $null  # @{ mm=; ss=; ff=; }


    $hdrList = New-Object System.Collections.ArrayList
    foreach ($pair in $hdrRecs) { [void]$hdrList.Add($pair) }

    for ($i = 0; $i -lt $hdrList.Count; $i++) {
    $idx = $hdrList[$i][0]
    $rec = $hdrList[$i][1]
    if ($null -eq $rec) { break }
    $b4 = [int]$rec[3]

    if ($b4 -ge 1 -and $b4 -le 99) {
        # At the first per-mylane record, look back two records ---
        if (-not $startedmylanes) {
        if ($i -ge 2) {
            $prev1 = $hdrList[$i-1][1]  # record right before first mylane => total length
            $prev2 = $hdrList[$i-2][1]  # record before that => number of mylanes

            # Number of mylanes is at B9 (position 9) of prev2
            $nummylanes = [int]$prev2[8]

            # Total disc length mm:ss:ff is at B9..B11 of prev1
            $discTotal = @{
            mm = [int]$prev1[8]
            ss = [int]$prev1[9]
            ff = [int]$prev1[10]
            }
        }
        }
    

        $startedmylanes = $true
        $mylaneNo = $b4
        $mm = [int]$rec[8]; $ss = [int]$rec[9]; $ff = [int]$rec[10]
        $mylaneTimes[$mylaneNo] = @{ mm=$mm; ss=$ss; ff=$ff }
        $headerEnd = $idx + 11
        continue
    } else {
        # Non-mylane record; if we've already started collecting mylane records, this means header is over.
        if ($startedmylanes) { break }
    }
    }

    if (-not $startedmylanes -or $null -eq $headerEnd) {
    throw "Could not locate per-mylane header records with times."
    }

    #----------------------------- Parse body -----------------------------

    $records = Scan-BodyRecords -Buf $buf -StartIndex $headerEnd

    # Buffers to accumulate text
    $wholecollectionTitleBytes  = New-Object System.Collections.Generic.List[byte]
    $wholecollectionauthorBytes = New-Object System.Collections.Generic.List[byte]
    $mylaneTitleBytes  = @{} # mylaneNo -> List[byte]
    $permylaneauthorBytes = @{} # mylaneNo -> List[byte]

    foreach ($rec in $records) {
    switch ($rec.Type) {
        0x80 { # Title
        Process-Payload -Payload $rec.Data -Startmylane $rec.mylaneNo `
            -wholecollectionBuf $wholecollectionTitleBytes -mylaneBufMap $mylaneTitleBytes
        }
        0x81 { # author
        Process-Payload -Payload $rec.Data -Startmylane $rec.mylaneNo `
            -wholecollectionBuf $wholecollectionauthorBytes -mylaneBufMap $permylaneauthorBytes
        }
        default { }
    }
    }

    # Decode
    $wholecollectionTitle  = Decode-BytesToString -Bytes ($wholecollectionTitleBytes.ToArray()) -Encoding $enc -Sanitize:$Sanitize
    $wholecollectionauthor = Decode-BytesToString -Bytes ($wholecollectionauthorBytes.ToArray()) -Encoding $enc -Sanitize:$Sanitize

    # Convert hashtables of List[byte] to strings
    $mylaneTitles = @{}
    foreach ($kv in $mylaneTitleBytes.GetEnumerator()) {
    $t = [int]$kv.Key
    $mylaneTitles[$t] = Decode-BytesToString -Bytes ($kv.Value.ToArray()) -Encoding $enc -Sanitize:$Sanitize
    }
    $permylanePerformers = @{}
    foreach ($kv in $permylaneauthorBytes.GetEnumerator()) {
    $t = [int]$kv.Key
    $permylanePerformers[$t] = Decode-BytesToString -Bytes ($kv.Value.ToArray()) -Encoding $enc -Sanitize:$Sanitize
    }

    # Ensure titles exist for all mylanes; default "mylane NN"
    $mylaneNumbers = ($mylaneTimes.Keys | Sort-Object)
    foreach ($t in $mylaneNumbers) {
    if (-not $mylaneTitles.ContainsKey($t) -or [string]::IsNullOrWhiteSpace($mylaneTitles[$t])) {
        $mylaneTitles[$t] = ('mylane {0:00}' -f $t)
    }
    }

    #----------------------------- Build txt -----------------------------

    $sb = New-Object System.Text.StringBuilder
    $inputLeaf = [System.IO.Path]::GetFileName($InputPath)
    [void]$sb.AppendLine(('FILE "{0}" BINARY' -f $inputLeaf))

    if ($discTotal -ne $null) {
    [void]$sb.AppendLine(('  REM DISC_LENGTH {0:D2}:{1:D2}:{2:D2}' -f `
        [int]$discTotal.mm, [int]$discTotal.ss, [int]$discTotal.ff))
    }

    if ($wholecollectionTitle)  { [void]$sb.AppendLine(('  TITLE "{0}"' -f ($wholecollectionTitle.Replace('"','""')))) }
    if ($wholecollectionauthor) { [void]$sb.AppendLine(('  PERFORMER "{0}"' -f ($wholecollectionauthor.Replace('"','""')))) }

    foreach ($t in $mylaneNumbers) {
    $time = $mylaneTimes[$t]
    $mm = [int]$time.mm; $ss = [int]$time.ss; $ff = [int]$time.ff
    $tt = $mylaneTitles[$t]
    [void]$sb.AppendLine(('  mylane {0:00} AUDIO' -f $t))
    [void]$sb.AppendLine(('    TITLE "{0}"' -f ($tt.Replace('"','""'))))
    if ($permylanePerformers.ContainsKey($t) -and -not [string]::IsNullOrWhiteSpace($permylanePerformers[$t])) {
        [void]$sb.AppendLine(('    PERFORMER "{0}"' -f ($permylanePerformers[$t].Replace('"','""'))))
    }
    [void]$sb.AppendLine(('    INDEX 01 {0:D2}:{1:D2}:{2:D2}' -f $mm,$ss,$ff))
    }

    [System.IO.File]::WriteAllText($OutputPath, $sb.ToString(), [System.Text.Encoding]::UTF8)
    Write-Host "txt written to: $OutputPath"
}  # End of etc-To-txt 


# ========== MAIN ==============


#Create output folder if it does not exist
if (-not [string]::IsNullOrWhiteSpace($OutputFolder)) {
    New-Item -ItemType Directory -Path $OutputFolder -Force | Out-Null
}

switch ($true) {
    {Test-Path -LiteralPath $InputFileOrFolder -PathType Container} 
        { if ($Recurse) {
             $FileList = Get-ChildItem -Path $InputFileOrFolder -Filter $InputPattern -File -Recurse ; break 
        }
        else
        {
             $FileList = Get-ChildItem -Path $InputFileOrFolder -Filter $InputPattern -File  ; break 
        }
    }
    {Test-Path -LiteralPath $InputFileOrFolder -PathType Leaf} 
        {   $FileList = Get-Item -LiteralPath $InputFileOrFolder; break}
    Default {
        Return "No file or folder found at $InputFileOrFolder"
    }
}

$FileList | ForEach-Object {

    # If output folder is not provided or is blank, use input file's folder

    $inPath  = $_.FullName
    $inFolder = split-path -path $_.FullName -Parent
    $inFileNameNoExt = $_.BaseName

    if ([string]::IsNullOrWhiteSpace($OutputFolder)) {
         $outPath = Join-path -path $inFolder -childpath ($inFileNameNoExt + ".bin")
    }
    Elseif (Test-Path -LiteralPath $OutputFolder -PathType Container) {
         $outPath = Join-path -path $OutputFolder -childpath ($inFileNameNoExt + ".bin")
    }
    Elseif (Test-Path -LiteralPath $OutputFolder -PathType Leaf) {
         $outPath = $OutputFolder
         if ([System.IO.Path]::GetExtension($outpath) -ne ".bin") {
            $outPath += ".bin"
        }
    }
    Else {
        write-host "Failed to find output location $OutputFolder"
    }

    $inPath  = $_.FullName
    $inFolder = split-path -path $_.FullName -Parent
    $inFileNameNoExt = $_.BaseName
 #   $inPathItem = (Get-Item $inPath)
    $fileSize = (Get-Item $inPath).Length
 #   $inPathItem.close()
 #   $inPathItem.dispose()
  
    # Convert 255 KB to bytes
    $threshold = 255KB

    # Check conditions
    if ($fileSize -eq 0) {
        continue
    } elseif ($fileSize -lt $threshold) {
         $outPath = $outPath -replace "\.[^.]+$", ".txt"
        if ($Force) {Convert-etcTotxt -InputPath $inPath -OutputPath $outPath -Force}
        Else {Convert-etcTotxt -InputPath $inPath -OutputPath $outPath}
    } else {
        if ($Force) {Convert-etcToBin -inPath $inPath -outPath $outPath -force}
        Else {Convert-etcToBin -inPath $inPath -outPath $outPath}
    }

}  # End ForEach File

}    # End of Function Convert-etcToBinOrtxt 

# Convert-Etoqso -InputFileOrFolder "E:\Enoughetctoqso\Sample1\Enough image" -OutputFolder "E:\Enoughetctoqso\Sample1\Enough image" 

#Convert-etcToBinOrtxt -InputFileOrFolder "E:\Sample1" -Recurse